function snorm = powmeth(A,P,Q,its)
%POWMETH  2-norm accuracy of an approx. to a matrix.
%
%
%   snorm = powmeth(A,P,Q)  computes an estimate snorm of the
%           spectral norm (the operator norm induced by the Euclidean
%           vector norm) of A-PQ, using 20 iterations of the power
%           method started with a random vector.
%
%   snorm = powmeth(A,P,Q,its)  computes an estimate snorm of the
%           spectral norm (the operator norm induced by the Euclidean
%           vector norm) of A-PQ, using its iterations of the power
%           method started with a random vector; its must be a positive
%           integer.
%
%
%   Increasing its improves the accuracy of the estimate snorm of the
%   spectral norm of A-PQ.
%
%
%   Note: POWMETH invokes RANDN. To obtain repeatable results, reset
%         the seed for the pseudorandom number generator.
%
%
%   inputs (the first three are required):
%   A -- first matrix in A-PQ whose spectral norm is being estimated
%   P -- second matrix in A-PQ whose spectral norm is being estimated
%   Q -- third matrix in A-PQ whose spectral norm is being estimated
%   its -- number of iterations of the power method to conduct;
%          its must be a positive integer, and defaults to 20
%
%   output:
%   snorm -- an estimate of the spectral norm of A-PQ (the estimate
%            fails to be accurate with exponentially low probability
%            as its increases; see references 1 and 2 below)
%
%
%   Example:
%     A = rand(1000,2)*rand(2,1000);
%     A = A/normest(A);
%     [P,Q] = als(A,2,true);
%     powmeth(A,P,Q)
%
%     This example produces a rank-4 approximation PQ to A, with
%     spectral-norm accuracy near the best possible for a rank-2
%     approximation.
%     powmeth(A,P,Q) outputs an estimate of the spectral norm
%     of A-PQ, which should be close to the machine precision.
%
%
%   References:
%   [1] Jacek Kuczynski and Henryk Wozniakowski, Estimating the largest
%       eigenvalues by the power and Lanczos methods with a random
%       start, SIAM Journal on Matrix Analysis and Applications, 13 (4):
%       1094-1122, 1992.
%   [2] Edo Liberty, Franco Woolfe, Per-Gunnar Martinsson, Vladimir
%       Rokhlin, and Mark Tygert, Randomized algorithms for the low-rank
%       approximation of matrices, Proceedings of the National Academy
%       of Sciences (USA), 104 (51): 20167-20172, 2007. (See the
%       appendix.)
%   [3] Franco Woolfe, Edo Liberty, Vladimir Rokhlin, and Mark Tygert,
%       A fast randomized algorithm for the approximation of matrices,
%       Applied and Computational Harmonic Analysis, 25 (3): 335-366,
%       2008. (See Section 3.4.)
%
%
%   See also POWMETHC, ALS, NORMEST, NORM.
%


%
% Check the number of inputs.
%
if(nargin < 3)
  error('MATLAB:powmeth:TooFewIn',...
        'There must be at least 3 inputs.')
end

if(nargin > 4)
  error('MATLAB:powmeth:TooManyIn',...
        'There must be at most 4 inputs.')
end

%
% Check the number of outputs.
%
if(nargout > 1)
  error('MATLAB:powmeth:TooManyOut',...
        'There must be at most 1 output.')
end

%
% Set the input its to its default value 20, if necessary.
%
if(nargin == 3)
  its = 20;
end

%
% Check the input arguments.
%
if(~isfloat(A))
  error('MATLAB:powmeth:In1NotFloat',...
        'Input 1 must be a floating-point matrix.')
end

if(isempty(A))
  error('MATLAB:powmeth:In1Empty',...
        'Input 1 must not be empty.')
end

if(~isfloat(P))
  error('MATLAB:powmeth:In2NotFloat',...
        'Input 2 must be a floating-point matrix.')
end

if(~isfloat(Q))
  error('MATLAB:powmeth:In3NotFloat',...
        'Input 3 must be a floating-point matrix.')
end

if(size(its,1) ~= 1 || size(its,2) ~= 1)
  error('MATLAB:powmeth:In4Not1x1',...
        'Input 4 must be a scalar.')
end

if(~(its > 0))
  error('MATLAB:powmeth:In4NonPos',...
        'Input 4 must be > 0.')
end

%
% Retrieve the dimensions of A, P, and Q.
%
[m n] = size(A);
[m2 k] = size(P);
[k2 n2] = size(Q);

%
% Make sure that the dimensions of A, P, and Q are commensurate.
%
if(m ~= m2)
  error('MATLAB:powmeth:In1In2BadDim',...
        'The 1st dims. of Inputs 1 and 2 must be equal.')
end

if(k ~= k2)
  error('MATLAB:powmeth:In2In3BadDim',...
        'The 2nd dim. of Input 2 must equal the 1st dim. of Input 3.')
end

if(n ~= n2)
  error('MATLAB:powmeth:In1In3BadDim',...
        'The 2nd dim. of Input 1 must equal the 2nd dim. of Input 3.')
end


if(m >= n)

%
% Generate a random vector x.
%
  if(isreal(A) && isreal(P) && isreal(Q))
    x = randn(n,1);
  end

  if(~isreal(A) || ~isreal(P) || ~isreal(Q))
    x = randn(n,1) + i*randn(n,1);
  end

  x = x/norm(x);

%
% Run its iterations of the power method, starting with the random x.
%
  for it = 1:its
%
%   Set y = (A-PQ)x.
%
    y = Q*x;
    y = P*y;
    y = A*x-y;
%
%   Set x = (A'-Q'P')y.
%
    x = (y'*P)';
    x = (x'*Q)';
    x = (y'*A)'-x;
%
%   Normalize x, memorizing its Euclidean norm.
%
    snorm = norm(x);
    if(snorm == 0)
      break;
    end
    x = x/snorm;
  end

  snorm = sqrt(snorm);

  clear x y;

end


if(m < n)

%
% Generate a random vector y.
%
  if(isreal(A) && isreal(P) && isreal(Q))
    y = randn(1,m);
  end

  if(~isreal(A) || ~isreal(P) || ~isreal(Q))
    y = randn(1,m) + i*randn(1,m);
  end

  y = y/norm(y);

%
% Run its iterations of the power method, starting with the random y.
%
  for it = 1:its
%
%   Set x = y(A-PQ).
%
    x = y*P;
    x = x*Q;
    x = y*A-x;
%
%   Set y = x(A'-Q'P').
%
    y = (Q*x')';
    y = (P*y')';
    y = (A*x')'-y;
%
%   Normalize y, memorizing its Euclidean norm.
%
    snorm = norm(y);
    if(snorm == 0)
      break;
    end
    y = y/snorm;
  end

  snorm = sqrt(snorm);

  clear x y;

end
