function [P,Q] = als(A,k,raw,its,l)
%ALS  Principal component analysis via alternating least squares
%
%
%   [P,Q] = ALS(A)  constructs an approximation PQ to C(A), with
%           spectral-norm accuracy near the best possible for a rank-6
%           approximation, where C(A) refers to A after centering its
%           columns, using 2 iterations of alternating least squares,
%           with block size 6+2=8, started with a min(m,n) x 8 random
%           matrix, when A is m x n; the reference below explains "near
%           the best possible." The smaller dimension of A must be >= 6
%           when A is the only input to ALS.
%
%   [P,Q] = ALS(A,k)  constructs an approximation PQ to C(A), with
%           spectral-norm accuracy near the best possible for a rank-k
%           approximation, where C(A) refers to A after centering its
%           columns, using 2 iterations of alternating least squares,
%           with block size k+2, started with a min(m,n) x (k+2)
%           random matrix, when A is m x n; the reference below
%           explains "near the best possible." k must be a positive
%           integer <= the smaller dimension of A.
%
%   [P,Q] = ALS(A,k,raw)  constructs an approximation PQ to A, with
%           spectral-norm accuracy near the best possible for a rank-k
%           approximation, centering the columns of A first when raw is
%           false, using 2 iterations of alternating least squares,
%           with block size k+2, started with a min(m,n) x (k+2) random
%           matrix, when A is m x n; the reference below explains "near
%           the best possible." k must be a positive integer <= the
%           smaller dimension of A.
%
%   [P,Q] = ALS(A,k,raw,its)  constructs an approximation PQ to A, with
%           spectral-norm accuracy near the best possible for a rank-k
%           approximation, centering the columns of A first when raw is
%           false, using its iterations of alternating least squares,
%           with block size k+2, started with a min(m,n) x (k+2) random
%           matrix, when A is m x n; the reference below explains "near
%           the best possible." k must be a positive integer <= the
%           smaller dimension of A, and its must be a nonnegative
%           integer.
%
%   [P,Q] = ALS(A,k,raw,its,l)  constructs an approximation PQ to A,
%           with spectral-norm accuracy near the best possible for a
%           rank-k approximation, centering the columns of A first when
%           raw is false, using its iterations of alternating least
%           squares, with block size l, started with a min(m,n) x l
%           random matrix, when A is m x n; the reference below
%           explains "near the best possible." k must be a positive
%           integer <= the smaller dimension of A, its must be a
%           nonnegative integer, and l must be a positive integer >= k.
%
%
%   P is m x l and Q is l x n, when A is m x n (and l defaults to k+2).
%
%   Increasing its or l improves the accuracy of the approximation PQ;
%   the reference below describes how the accuracy depends on its and l.
%   Please note that even its=1 guarantees superb accuracy, whether or
%   not there is any gap in the singular values of the matrix A being
%   approximated, at least when measuring accuracy as the spectral norm
%   ||A-PQ|| of A-PQ (relative to the spectral norm ||A|| of A).
%
%
%   Note: ALS invokes RAND. To obtain repeatable results, reset the seed
%         for the pseudorandom number generator.
%
%   Note: The user may ascertain the accuracy of the approximation PQ
%         to A by invoking POWMETH(A,P,Q), when raw is true.
%         The user may ascertain the accuracy of the approximation PQ
%         to C(A), where C(A) refers to A after centering its columns,
%         by invoking POWMETHC(A,P,Q), when raw is false.
%
%
%   inputs (the first is required):
%   A -- matrix being approximated
%   k -- rank of the best approximation being approximated;
%        k must be a positive integer <= the smaller dimension of A,
%        and defaults to 6
%   raw -- centers A when raw is false but does not when raw is true;
%          raw must be a Boolean and defaults to false
%   its -- number of iterations of alternating least squares to do;
%          its must be a nonnegative integer, and defaults to 2
%   l -- block size of the alternating least squares iterations;
%        l must be a positive integer >= k, and defaults to k+2
%
%   outputs (both are required):
%   P -- m x l matrix in the rank-l approximation PQ to A or C(A),
%        where A is m x n, and C(A) refers to A after centering its
%        columns
%   Q -- l x n matrix in the rank-l approximation PQ to A or C(A),
%        where A is m x n, and C(A) refers to A after centering its
%        columns
%
%
%   Example:
%     A = rand(1000,2)*rand(2,1000);
%     A = A/normest(A);
%     [P,Q] = als(A,2,true);
%     powmeth(A,P,Q)
%
%     This example produces a rank-4 approximation PQ to A, with
%     spectral-norm accuracy near the best possible for a rank-2
%     approximation.
%     powmeth(A,P,Q) outputs an estimate of the spectral norm
%     of A-PQ, which should be close to the machine precision.
%
%
%   Reference:
%   Nathan Halko, Per-Gunnar Martinsson, and Joel Tropp,
%   Finding structure with randomness: probabilistic algorithms
%   for constructing approximate matrix decompositions,
%   arXiv:0909.4061 [math.NA; math.PR], 2009
%   (available at http://arxiv.org).
%
%
%   See also POWMETH, POWMETHC, PCACOV, PRINCOMP, SVDS.
%



%
% Check the number of inputs.
%
if(nargin < 1)
  error('MATLAB:als:TooFewIn',...
        'There must be at least 1 input.')
end

if(nargin > 5)
  error('MATLAB:als:TooManyIn',...
        'There must be at most 5 inputs.')
end

%
% Check the number of outputs.
%
if(nargout ~= 2)
  error('MATLAB:als:WrongNumOut',...
        'There must be exactly 2 outputs.')
end

%
% Set the inputs k, raw, its, and l to default values, if necessary.
%
if(nargin == 1)
  k = 6;
  raw = false;
  its = 2;
  l = k+2;
end

if(nargin == 2)
  raw = false;
  its = 2;
  l = k+2;
end

if(nargin == 3)
  its = 2;
  l = k+2;
end

if(nargin == 4)
  l = k+2;
end

%
% Check the first input argument.
%
if(~isfloat(A))
  error('MATLAB:als:In1NotFloat',...
        'Input 1 must be a floating-point matrix.')
end

if(isempty(A))
  error('MATLAB:als:In1Empty',...
        'Input 1 must not be empty.')
end

%
% Retrieve the dimensions of A.
%
[m n] = size(A);

%
% Check the remaining input arguments.
%
if(size(k,1) ~= 1 || size(k,2) ~= 1)
  error('MATLAB:als:In2Not1x1',...
        'Input 2 must be a scalar.')
end

if(size(raw,1) ~= 1 || size(raw,2) ~= 1)
  error('MATLAB:als:In3Not1x1',...
        'Input 3 must be a scalar.')
end

if(size(its,1) ~= 1 || size(its,2) ~= 1)
  error('MATLAB:als:In4Not1x1',...
        'Input 4 must be a scalar.')
end

if(size(l,1) ~= 1 || size(l,2) ~= 1)
  error('MATLAB:als:In5Not1x1',...
        'Input 5 must be a scalar.')
end

if(~(k > 0))
  error('MATLAB:als:In2NonPos',...
        'Input 2 must be > 0.')
end

if((nargin > 1) && ((k > m) || (k > n)))
  error('MATLAB:als:In2TooBig',...
        'Input 2 must be <= the smaller dimension of Input 1.')
end

if((nargin == 1) && ((k > m) || (k > n)))
  error('MATLAB:als:InTooTiny',...
        'The smaller dimension of the input must be >= 6.')
end

if(~(its >= 0))
  error('MATLAB:als:In4Bad',...
        'Input 4 must be >= 0.')
end

if(l < k)
  error('MATLAB:als:In5ltIn2',...
        'Input 5 must be >= Input 2.')
end

%
% Reduce l appropriately, if necessary.
%
if(l > m)
  l = m
end
if(l > n)
  l = n
end



if(raw)


  if(m >= n)

%
%   Apply the pseudoinverse of a random matrix to A', obtaining P'.
%
    if(isreal(A))
      Q = 2*rand(l,n)-ones(l,n);
    end

    if(~isreal(A))
      Q = (2*rand(l,n)-ones(l,n)) + i*(2*rand(l,n)-ones(l,n));
    end

    P = (Q'\A')';

%
%   Conduct iterations of alternating least squares.
%
    for it = 1:its

      Q = P\A;
      P = (Q'\A')';

    end

    Q = P\A;

  end


  if(m < n)

%
%   Apply the pseudoinverse of a random matrix to A, obtaining Q.
%
    if(isreal(A))
      P = 2*rand(m,l)-ones(m,l);
    end

    if(~isreal(A))
      P = (2*rand(m,l)-ones(m,l)) + i*(2*rand(m,l)-ones(m,l));
    end

    Q = P\A;

%
%   Conduct iterations of alternating least squares.
%
    for it = 1:its

      P = (Q'\A')';
      Q = P\A;

    end

    P = (Q'\A')';

  end


end



if(~raw)


%
% Calculate the average of the entries in every column.
%
  c = sum(A)/m;


  if(m >= n)

%
%   Apply the pseudoinverse of a random matrix to the centered A',
%   obtaining P'.
%
    if(isreal(A))
      Q = 2*rand(l,n)-ones(l,n);
    end

    if(~isreal(A))
      Q = (2*rand(l,n)-ones(l,n)) + i*(2*rand(l,n)-ones(l,n));
    end

    P = (Q'\A')' - ones(m,1)*(Q'\c')';

%
%   Conduct iterations of alternating least squares.
%
    for it = 1:its

      Q = P\A - (P\ones(m,1))*c;
      P = (Q'\A')' - ones(m,1)*(Q'\c')';

    end

    Q = P\(A-ones(m,1)*c);


  end


  if(m < n)

%
%   Apply the pseudoinverse of a random matrix to the centered A,
%   obtaining Q.
%
    if(isreal(A))
      P = 2*rand(m,l)-ones(m,l);
    end

    if(~isreal(A))
      P = (2*rand(m,l)-ones(m,l)) + i*(2*rand(m,l)-ones(m,l));
    end

    Q = P\A - (P\ones(m,1))*c;

%
%   Conduct iterations of alternating least squares.
%
    for it = 1:its

      P = (Q'\A')' - ones(m,1)*(Q'\c')';
      Q = P\A - (P\ones(m,1))*c;

    end

    P = (Q'\(A'-c'*ones(1,m)))';


  end


end
